<?php
/**
 * ===========================================
 * FLOWBOT DCI - MAIN CONFIGURATION
 * ===========================================
 * Loads environment variables and provides configuration
 */

declare(strict_types=1);

// Load environment variables from .env file
$envFile = dirname(__DIR__) . '/.env';
$envLoaded = false;
$envErrors = [];

if (file_exists($envFile)) {
    if (is_readable($envFile)) {
        $lines = @file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if ($lines !== false) {
            foreach ($lines as $line) {
                $line = trim($line);
                if (empty($line) || strpos($line, '#') === 0) continue;
                if (strpos($line, '=') === false) continue;

                $parts = explode('=', $line, 2);
                if (count($parts) !== 2) continue;

                $name = trim($parts[0]);
                $value = trim($parts[1]);

                // Remove quotes if present
                if ((substr($value, 0, 1) === '"' && substr($value, -1) === '"') ||
                    (substr($value, 0, 1) === "'" && substr($value, -1) === "'")) {
                    $value = substr($value, 1, -1);
                }

                if (!array_key_exists($name, $_ENV)) {
                    $_ENV[$name] = $value;
                    putenv("$name=$value");
                }
            }
            $envLoaded = true;
        } else {
            $envErrors[] = "Cannot read .env file contents";
        }
    } else {
        $envErrors[] = ".env file is not readable";
    }
} else {
    $envErrors[] = ".env file not found at: {$envFile}";
}

// Store env loading status for debugging
$GLOBALS['env_loaded'] = $envLoaded;
$GLOBALS['env_errors'] = $envErrors;

/**
 * Get environment variable with default fallback
 * Using function_exists to prevent redeclaration errors
 */
if (!function_exists('env')) {
    function env(string $key, $default = null) {
        $value = $_ENV[$key] ?? getenv($key);
        if ($value === false || $value === null || $value === '') {
            return $default;
        }

        // Convert string booleans
        $lowerValue = strtolower((string)$value);
        if ($lowerValue === 'true') return true;
        if ($lowerValue === 'false') return false;
        if ($lowerValue === 'null') return null;

        // Convert numeric strings
        if (is_numeric($value)) {
            return strpos($value, '.') !== false ? (float)$value : (int)$value;
        }

        return $value;
    }
}

// Application configuration array
return [
    'app' => [
        'debug' => env('APP_DEBUG', false),
        'timezone' => env('APP_TIMEZONE', 'America/Sao_Paulo'),
    ],

    'database' => [
        'host' => env('DB_HOST', 'localhost'),
        'name' => env('DB_NAME', 'digupdog_FEED'),
        'user' => env('DB_USER', 'digupdog_FEEDadmin'),
        'password' => env('DB_PASSWORD', ''),
        'charset' => env('DB_CHARSET', 'utf8mb4'),
    ],

    // v2.1: Less aggressive phases to avoid triggering anti-bot detection
    // Phase 0: Gentle start to avoid flagging IP
    // Phase 1-3: Progressively more patient with longer timeouts
    'phases' => [
        ['concurrency' => (int)env('PHASE_0_CONCURRENCY', 50),  'timeout' => (int)env('PHASE_0_TIMEOUT', 3)],
        ['concurrency' => (int)env('PHASE_1_CONCURRENCY', 30),  'timeout' => (int)env('PHASE_1_TIMEOUT', 5)],
        ['concurrency' => (int)env('PHASE_2_CONCURRENCY', 15),  'timeout' => (int)env('PHASE_2_TIMEOUT', 8)],
        ['concurrency' => (int)env('PHASE_3_CONCURRENCY', 5),   'timeout' => (int)env('PHASE_3_TIMEOUT', 15)],
    ],

    'processing' => [
        'max_links_per_batch' => (int)env('MAX_LINKS_PER_BATCH', 10),
        'temp_folder_lifetime' => (int)env('TEMP_FOLDER_LIFETIME', 86400),
        'ssl_verify' => env('SSL_VERIFY', true),
    ],

    'defaults' => [
        'thumbnail' => env('DEFAULT_THUMBNAIL', 'fallback_image.jpg'),
        'favicon' => env('DEFAULT_FAVICON', 'default_favicon.ico'),
        'profile_picture' => env('DEFAULT_PROFILE_PICTURE', 'https://example.com/default_profile.jpg'),
        'user_password' => env('DEFAULT_USER_PASSWORD', 'ChangeMe123!'),
    ],

    'paths' => [
        'temp' => dirname(__DIR__) . '/temp',
        'views' => dirname(__DIR__) . '/views',
        'public' => dirname(__DIR__) . '/public',
    ],
];
